<?php
/**
 * WP CLI commands for test purpose
 *
 * @author  YITH <plugins@yithemes.com>
 * @package YITH\AjaxProductFilter\Classes\Tools
 * @version 4.0.3
 */

if ( ! defined( 'YITH_WCAN' ) ) {
	exit;
} // Exit if accessed directly

use function WP_CLI\Utils\make_progress_bar;

if ( ! class_exists( 'YITH_WCAN_CLI_Test_Commands' ) ) {
	/**
	 * WP-CLI Integration class
	 */
	class YITH_WCAN_CLI_Test_Commands extends WP_CLI_Command {
		/**
		 * Generate products.
		 *
		 * ## OPTIONS
		 *
		 * <amount>
		 * : The amount of products to generate
		 * ---
		 * default: 100
		 * ---
		 *
		 * ## EXAMPLES
		 * yith-wcan generate products 100
		 *
		 * @param array $args       Arguments specified.
		 * @param array $assoc_args Associative arguments specified.
		 *
		 * @throws \WP_CLI\ExitException Exception Throws expcetion on failure.
		 */
		public static function generate_products( $args, $assoc_args ) {
			list( $amount ) = $args;

			$time_start = microtime( true );

			WP_CLI::line( 'Initializing...' );

			if ( ! defined( 'YITH_WCAN' ) ) {
				WP_CLI::error( 'YITH WooCommerce Ajax Product filter is not active!', true );
			}

			$include_variables   = ! empty( $assoc_args['include_variables'] );
			$generate_attributes = ! empty( $assoc_args['generate_attributes'] );
			$progress            = make_progress_bar( 'Generating products', $amount );
			$generated           = 0;

			if ( $generate_attributes ) {
				self::generate_attributes();
			}

			for ( $i = 1; $i <= $amount; $i ++ ) {
				$should_be_variable = $include_variables && self::check_your_luck();

				$price      = wp_rand( 0, 100 );
				$sale_price = self::check_your_luck() ? wp_rand( 0, $price ) : '';
				$props      = array(
					'name'          => 'Generated product ' . ucfirst( self::get_random_string() ),
					'regular_price' => $price,
					'sale_price'    => $sale_price,
					'stock_status'  => self::check_your_luck( 0.95 ) ? 'instock' : 'outofstock',
				);

				if ( ! class_exists( 'WC_Helper_Product' ) ) {
					include_once WC()->plugin_path() . '/tests/legacy/framework/helpers/class-wc-helper-product.php';
				}

				if ( ! $should_be_variable ) {
					// for simple product, just keave the task to WC.
					$product = WC_Helper_Product::create_simple_product( true, $props );
				} else {
					// in case product is variable, we want more granular control.
					$product = new WC_Product_Variable();
					$product->set_props( $props );

					// save before generating variations.
					$product->save();

					// generate variations.
					self::set_product_attributes( $product );
					self::create_product_variations( $product );
				}

				// assigN terms.
				self::set_product_terms( $product );

				$product->update_meta_data( 'created_via', 'YITH WCAN Generator' );
				$product->save();

				$progress->tick();
				$generated ++;
			}

			$time_end       = microtime( true );
			$execution_time = round( ( $time_end - $time_start ), 2 );
			$display_time   = self::get_display_time( $execution_time );

			$progress->finish();

			WP_CLI::success( $generated . ' products generated in ' . $display_time );
		}

		/**
		 * Create random attributes to use for products
		 *
		 * ## OPTIONS
		 *
		 * <amount>
		 * : The amount of products to generate
		 * ---
		 * default: 100
		 * ---
		 *
		 * ## EXAMPLES
		 * yith-wcan generate attributes 100
		 *
		 * @param array $args Arguments specified.
		 *
		 * @return void.
		 */
		public static function generate_attributes( $args = false ) {
			if ( $args ) {
				list( $amount ) = $args;
			} else {
				$amount = wp_rand( 1, 3 );
			}

			for ( $i = 0; $i < $amount; $i ++ ) {
				$attribute_name  = 'Generated attribute ' . ucfirst( self::get_random_string() );
				$number_of_terms = wp_rand( 3, 10 );
				$terms           = array();

				for ( $k = 0; $k < $number_of_terms; $k ++ ) {
					$terms[] = 'Attr ' . self::get_random_string( 5 );
				}

				WP_CLI::line( 'Generating ' . $attribute_name );

				WC_Helper_Product::create_product_attribute_object( $attribute_name, $terms );
			}
		}

		/**
		 * Delete products generated by this tool
		 *
		 * ## EXAMPLES
		 * yith-wcan delete products
		 *
		 * @return void.
		 */
		public static function delete_products() {
			$time_start = microtime( true );

			add_filter(
				'woocommerce_product_data_store_cpt_get_products_query',
				function ( $query, $query_vars ) {
					if ( ! empty( $query_vars['yith_wcan_test'] ) ) {
						$query['meta_query'][] = array(
							'key'   => 'created_via',
							'value' => 'YITH WCAN Generator',
						);
					}

					return $query;
				},
				10,
				2
			);

			$products_to_delete = wc_get_products(
				array(
					'limit'          => - 1,
					'return'         => 'ids',
					'yith_wcan_test' => true,
				)
			);

			$progress = make_progress_bar( 'Deleting products', count( $products_to_delete ) );
			$deleted  = 0;

			foreach ( $products_to_delete as $product_id ) {
				wp_delete_post( $product_id );
				$progress->tick();
				$deleted ++;
			}

			$progress->finish();

			$time_end       = microtime( true );
			$execution_time = round( ( $time_end - $time_start ), 2 );
			$display_time   = self::get_display_time( $execution_time );

			WP_CLI::success( $deleted . ' products deleted in ' . $display_time );
		}

		/**
		 * Switch products generated by this tool to published
		 *
		 * ## EXAMPLES
		 * yith-wcan publish products
		 *
		 * @return void.
		 */
		public static function publish_products() {
			$time_start = microtime( true );

			add_filter(
				'woocommerce_product_data_store_cpt_get_products_query',
				function ( $query, $query_vars ) {
					if ( ! empty( $query_vars['yith_wcan_test'] ) ) {
						$query['meta_query'][] = array(
							'key'   => 'created_via',
							'value' => 'YITH WCAN Generator',
						);
					}

					return $query;
				},
				10,
				2
			);

			$products = wc_get_products(
				array(
					'limit'          => - 1,
					'return'         => 'ids',
					'yith_wcan_test' => true,
				)
			);

			$progress = make_progress_bar( 'Updating products', count( $products ) );
			$updated  = 0;

			foreach ( $products as $product_id ) {
				self::switch_product_status( $product_id, 'publish' );
				$progress->tick();
				$updated ++;
			}

			$progress->finish();

			$time_end       = microtime( true );
			$execution_time = round( ( $time_end - $time_start ), 2 );
			$display_time   = self::get_display_time( $execution_time );

			WP_CLI::success( $updated . ' products updated in ' . $display_time );
		}

		/**
		 * Switch products generated by this tool to draft
		 *
		 * ## EXAMPLES
		 * yith-wcan draft products
		 *
		 * @return void.
		 */
		public static function draft_products() {
			$time_start = microtime( true );

			add_filter(
				'woocommerce_product_data_store_cpt_get_products_query',
				function ( $query, $query_vars ) {
					if ( ! empty( $query_vars['yith_wcan_test'] ) ) {
						$query['meta_query'][] = array(
							'key'   => 'created_via',
							'value' => 'YITH WCAN Generator',
						);
					}

					return $query;
				},
				10,
				2
			);

			$products = wc_get_products(
				array(
					'limit'          => - 1,
					'return'         => 'ids',
					'yith_wcan_test' => true,
				)
			);

			$progress = make_progress_bar( 'Updating products', count( $products ) );
			$updated  = 0;

			foreach ( $products as $product_id ) {
				self::switch_product_status( $product_id );
				$progress->tick();
				$updated ++;
			}

			$progress->finish();

			$time_end       = microtime( true );
			$execution_time = round( ( $time_end - $time_start ), 2 );
			$display_time   = self::get_display_time( $execution_time );

			WP_CLI::success( $updated . ' products updated in ' . $display_time );
		}

		/**
		 * Change status of a batch of products during WP CLI executions
		 *
		 * @param int    $product_id Product id.
		 * @param string $status     Status to assigne to the post.
		 * @return int Return value of {@see wp_update_post}.
		 */
		protected static function switch_product_status( $product_id, $status = 'draft' ) {
			return wp_update_post(
				array(
					'ID'          => $product_id,
					'post_status' => $status,
				)
			);
		}

		/**
		 * Set attributes for a specific product, getting them randomly from existing ones
		 *
		 * @param WC_Product $product Product to modify.
		 *
		 * @return WC_Product
		 */
		protected static function set_product_attributes( $product ) {
			$attributes_taxonomies = wc_get_attribute_taxonomies();

			if ( empty( $attributes_taxonomies ) ) {
				return $product;
			}

			// pickup a couple of attributes from the array, and populate attributes array.
			$num_of_taxonomies   = count( $attributes_taxonomies );
			$num_of_attributes   = wp_rand( 1, $num_of_taxonomies );
			$selected_attributes = array();

			for ( $i = 0; $i < $num_of_attributes; $i ++ ) {
				shuffle( $attributes_taxonomies );

				$selected_tax = $attributes_taxonomies[0];
				$taxonomy     = 'pa_' . $selected_tax->attribute_name;

				$terms = get_terms(
					array(
						'taxonomy'   => $taxonomy,
						'limit'      => 10,
						'fields'     => 'ids',
						'hide_empty' => false,
					)
				);

				shuffle( $terms );

				$selected_terms = array_slice( $terms, 0, 3 );
				$attribute      = new WC_Product_Attribute();

				$attribute->set_id( $selected_tax->attribute_id );
				$attribute->set_name( $taxonomy );
				$attribute->set_options( $selected_terms );
				$attribute->set_position( 1 );
				$attribute->set_visible( true );
				$attribute->set_variation( 0 === $i || self::check_your_luck() ); // always set at least one attribute used for variation.

				$selected_attributes[] = $attribute;
			}

			// save attributes for the product.
			$product->set_attributes( $selected_attributes );

			return $product;
		}

		/**
		 * Create random variations for any of the attributes set for the product
		 *
		 * @param WC_Product $product Product to modify.
		 *
		 * @return WC_Product
		 */
		protected static function create_product_variations( $product ) {
			// chose a random attribute from the main product.
			$attributes = $product->get_attributes();
			$variations = array();

			// select only attributes used for variations.
			$variation_attributes = array_filter(
				$attributes,
				function ( $item ) {
					return $item->get_variation();
				}
			);

			$num_of_variations = wp_rand( 1, 10 );
			$product_price     = (float) $product->get_regular_price();
			$already_processed = array();

			for ( $i = 0; $i < $num_of_variations; $i ++ ) {
				$difference = wp_rand( 0, 15 ) * $product_price / 100;
				$price      = $product_price + ( ( self::check_your_luck() ? 1 : - 1 ) * $difference );

				$selected_attributes = array();

				foreach ( $variation_attributes as $attribute ) {
					$terms = get_terms(
						array(
							'taxonomy'   => $attribute->get_taxonomy(),
							'hide_empty' => false,
							'include'    => $attribute->get_options(),
							'fields'     => 'slugs',
						)
					);
					shuffle( $terms );

					$selected_attributes[ $attribute->get_name() ] = $terms[0];
				}

				$hash = md5( http_build_query( $selected_attributes ) );

				if ( in_array( $hash, $already_processed, true ) ) {
					continue;
				}

				$already_processed[] = $hash;

				$variations[] = WC_Helper_Product::create_product_variation_object(
					$product->get_id(),
					self::get_random_string(),
					$price,
					$selected_attributes
				);
			}

			$variation_ids = array_map(
				function ( $variation ) {
					return $variation->get_id();
				},
				$variations
			);

			// save product with new variations.
			$product->set_children( $variation_ids );
			$product->save();

			return $product;
		}

		/**
		 * Sets category/tags (and any other product taxonomy) for the product we're creating
		 *
		 * @param WC_Product $product Product to modify.
		 *
		 * @return WC_Product
		 */
		protected static function set_product_terms( $product ) {
			$taxonomies = get_object_taxonomies( 'product', 'objects' );

			if ( empty( $taxonomies ) ) {
				return $product;
			}

			foreach ( $taxonomies as $taxonomy_slug => $taxonomy ) {
				if ( 0 === strpos( $taxonomy_slug, 'pa_' ) ) {
					continue;
				}

				$terms = get_terms(
					array(
						'taxonomy'   => $taxonomy_slug,
						'limit'      => 10,
						'hide_empty' => false,
						'fields'     => 'ids',
					)
				);

				shuffle( $terms );

				$number_of_terms = wp_rand( 1, count( $terms ) );
				$selected_terms  = array_slice( $terms, 0, $number_of_terms );

				wp_set_post_terms( $product->get_id(), $selected_terms, $taxonomy_slug, true );
			}

			return $product;
		}

		/**
		 * Randomly returns true or false
		 *
		 * @param float $chances Chances of a positive result, in float from 0 (impossible) to 1 (sure).
		 *
		 * @return bool Whether you're lucky or not.
		 */
		protected static function check_your_luck( $chances = 0.5 ) {
			return wp_rand( 0, 1 ) > $chances;
		}

		/**
		 * Retrieve a display time
		 *
		 * @param int $seconds Seconds.
		 *
		 * @return string
		 */
		public static function get_display_time( $seconds ) {
			$display_time = $seconds . ' seconds';
			if ( $seconds >= 60 ) {
				$human_time   = human_time_diff( 0, $seconds );
				$display_time = $human_time . " ($display_time)";
			}

			return $display_time;
		}

		/**
		 * Generate random string
		 *
		 * @param int $length Length of the string to generate.
		 *
		 * @return string Generated string.
		 */
		protected static function get_random_string( $length = 10 ) {
			$alphabet = 'abcdefghijklmnopqrstuwxyz0123456789';
			$string   = '';

			for ( $i = 0; $i < $length; $i ++ ) {
				$index = wp_rand( 0, strlen( $alphabet ) - 1 );

				$string .= $alphabet[ $index ];
			}

			return $string;
		}
	}

	try {
		WP_CLI::add_command(
			'yith-wcan-test generate products',
			array( 'YITH_WCAN_CLI_Test_Commands', 'generate_products' ),
			array(
				'synopsis' => array(
					array(
						'name'     => 'amount',
						'type'     => 'positional',
						'optional' => true,
						'default'  => 100,
					),
					array(
						'name'     => 'include_variables',
						'type'     => 'flag',
						'optional' => true,
					),
					array(
						'name'     => 'generate_attributes',
						'type'     => 'flag',
						'optional' => true,
					),
				),
			)
		);

		WP_CLI::add_command(
			'yith-wcan-test generate attributes',
			array( 'YITH_WCAN_CLI_Test_Commands', 'generate_attributes' ),
			array(
				'synopsis' => array(
					array(
						'name'     => 'amount',
						'type'     => 'positional',
						'optional' => true,
						'default'  => 100,
					),
				),
			)
		);

		WP_CLI::add_command(
			'yith-wcan-test delete products',
			array( 'YITH_WCAN_CLI_Test_Commands', 'delete_products' )
		);

		WP_CLI::add_command(
			'yith-wcan-test draft products',
			array( 'YITH_WCAN_CLI_Test_Commands', 'draft_products' )
		);

	} catch ( Exception $e ) {
		error_log( 'Error when declaring wp-cli commands: ' . $e->getMessage() ); // phpcs:ignore
	}
}
